//////////////////////////////////////////////////////////////////////////////
//
//  Copyright 2017 Autodesk, Inc.  All rights reserved.
//
//  Use of this software is subject to the terms of the Autodesk license 
//  agreement provided at the time of installation or download, or which 
//  otherwise accompanies this software in either electronic or hard copy form.   
//
//////////////////////////////////////////////////////////////////////////////

#include "resource.h"

#include "MAXtoATestObject.h"
#include "MAXtoATestObjectUI.h"

// arnold
#undef min
#undef max
#include <ai.h>

#include <ai_nodes.h>
#include <ai_array.h>

#include <iparamb2.h>
#include "IPathConfigMgr.h"

#include "Graphics/IDisplayManager.h"
#include <Graphics/Utilities/MeshEdgeRenderItem.h>
#include <Graphics/CustomRenderItemHandle.h>
#include <Graphics/RenderNodeHandle.h>
#include "AssetManagement/AssetUser.h"
#include "AssetManagement/iassetmanager.h"

#include "path.h"
#include "IFileResolutionManager.h"

#include "../../include/3dsmax_banned.h"

using namespace MaxSDK::AssetManagement;

#define DD (.01f)

namespace 
{
    void MakeQuad(Face *f, int a, int b , int c , int d, int sg) 
    {
        f[0].setVerts( a, b, c);
        f[0].setSmGroup(sg);
        f[0].setEdgeVisFlags(1,1,0);
        f[1].setVerts( c, d, a);
        f[1].setSmGroup(sg);
        f[1].setEdgeVisFlags(1,1,0);
    }
}

MAXtoATestObject_translationInterface::MAXtoATestObject_translationInterface(MAXtoATestObject* obj)
{
    m_obj = obj;
}

//==============================================================================
// class MAXtoATestObject_ClassDesc
//==============================================================================

class MAXtoATestObject_ClassDesc : public ClassDesc2 
{
public:

    MAXtoATestObject_ClassDesc();

    int IsPublic();
    void* Create(BOOL loading);
    const TCHAR* ClassName();
    SClass_ID SuperClassID();
    Class_ID ClassID();
    const TCHAR* Category();
    const TCHAR* InternalName();
    HINSTANCE HInstance();    

protected:

    virtual MaxSDK::QMaxParamBlockWidget* CreateQtWidget(
        ReferenceMaker& owner,
        IParamBlock2& paramBlock,
        const MapID paramMapID,
        MSTR& rollupTitle,
        int& rollupFlags,
        int& rollupCategory) override;

private:
    
    // Prefernces change callback
    static void PreferencesChangeCallback(void* param);

    bool m_preferenceCallbackRegistered;
};

MAXtoATestObject_ClassDesc::MAXtoATestObject_ClassDesc()
: m_preferenceCallbackRegistered(false)
{
}

MaxSDK::QMaxParamBlockWidget* MAXtoATestObject_ClassDesc::CreateQtWidget(
    ReferenceMaker& owner,
    IParamBlock2& paramBlock,
    const MapID paramMapID,
    MSTR& rollupTitle,
    int& rollupFlags,
    int& rollupCategory)
{
    switch (paramMapID)
    {
        case 0:
        {
            rollupTitle = L"MAXtoATest Object";
            MAXtoATestObjectUI* const widget = new MAXtoATestObjectUI(owner, paramBlock);
            rollupCategory -= 70;
            return widget;
        }
    }

    return nullptr;
}

void MAXtoATestObject_ClassDesc::PreferencesChangeCallback(void* param) 
{
    DbgAssert(param != NULL);
    MAXtoATestObject_ClassDesc* classDesc = static_cast<MAXtoATestObject_ClassDesc*>(param);
    if (DbgVerify(classDesc != nullptr))
    {
        ClassEntry* classEntry = GetCOREInterface()->GetDllDir().ClassDir().FindClassEntry(classDesc->SuperClassID(), classDesc->ClassID());
        if (DbgVerify(classEntry != NULL)) 
        {
            // Delete the class and add it back. This updates the IsPublic state and
            // also updates the class list in the create panel
            GetCOREInterface()->DeleteClass(classDesc);
            GetCOREInterface()->AddClass(classDesc);

            // Update the 'isPublic' state of the class entry. Do this by completely
            // updating the class entry
            //classEntry->Set(classDesc, classEntry->DllNumber(), classEntry->ClassNumber(), classEntry->IsLoaded());
        }
    }
}

int MAXtoATestObject_ClassDesc::IsPublic() 
{
    return TRUE;
}

void* MAXtoATestObject_ClassDesc::Create(BOOL loading) 
{
    return new MAXtoATestObject(loading != 0);
}

const TCHAR* MAXtoATestObject_ClassDesc::ClassName() 
{
    return L"MAXtoATestObject";
}

SClass_ID MAXtoATestObject_ClassDesc::SuperClassID() 
{
    return GEOMOBJECT_CLASS_ID;
}

Class_ID MAXtoATestObject_ClassDesc::ClassID() 
{
    return MAXtoATestOBJECT_CLASS_ID;
}

const TCHAR* MAXtoATestObject_ClassDesc::Category()
{
    return L"MAXtoATest";
}

const TCHAR* MAXtoATestObject_ClassDesc::InternalName() 
{
    return _T("MAXtoATest Object");
}

HINSTANCE MAXtoATestObject_ClassDesc::HInstance() 
{
    extern HINSTANCE hInstance;
    return hInstance;
}
 
//==============================================================================
// class MAXtoATestObject_CreateCallback
//
// The mouse creation callback for the Arnold Procedural object.
//==============================================================================

class MAXtoATestObject_CreateCallback : public CreateMouseCallBack 
{
public:

    static MAXtoATestObject_CreateCallback& GetInstance() { return m_theInstance; };

    void SetObject(MAXtoATestObject* object) { m_object = object; }

    // -- from CreateMouseCallBack
    int proc(ViewExp *vpt,int msg, int point, int flags, IPoint2 m, Matrix3& mat);

private:

    static MAXtoATestObject_CreateCallback m_theInstance;

    MAXtoATestObject_CreateCallback();

    // The current object
    MAXtoATestObject* m_object;

    // The initial point & snap point
    Point3 m_point;
    IPoint2 m_snapPoint;  
};

MAXtoATestObject_CreateCallback MAXtoATestObject_CreateCallback::m_theInstance;

MAXtoATestObject_CreateCallback::MAXtoATestObject_CreateCallback()
: m_object(NULL)
{
}

int MAXtoATestObject_CreateCallback::proc(ViewExp *vpt,int msg, int point, int flags, IPoint2 m, Matrix3& mat)
{
    if ( ! vpt || ! vpt->IsAlive() )
    {
        // why are we here
        DbgAssert(!_T("Doing proc() on invalid viewport!"));
        return FALSE;
    }
  
    DbgAssert(m_object != NULL);

    // Stole this code from DummyObjectCreateCallBack::proc(), see src\core\dummy.cpp

    switch(msg) 
    {
    case MOUSE_FREEMOVE:
#ifdef _OSNAP
        vpt->SnapPreview(m, m, NULL, SNAP_IN_3D);
#endif
        break;

    case MOUSE_POINT:
        m_object->ApplyUpAxis();
        // Fall through intentional
    case MOUSE_MOVE:
        switch(point) 
        {
        case 0:
            if(DbgVerify(m_object != nullptr))
            {
                m_object->SetBox(Box3(Point3(-DD, -DD, -DD), Point3(DD, DD, DD)));
                m_object->SetCreating(true);
    #ifdef _3D_CREATE
                m_point = vpt->SnapPoint(m, m, NULL, SNAP_IN_3D);
    #else
                m_point = vpt->SnapPoint(m, m, NULL, SNAP_IN_PLANE);
    #endif

                mat.SetTrans(m_point);
                m_snapPoint = m;
                m_object->InvalidateMesh();
            }
            break;

        case 1:
            {
                float l;
#ifdef _OSNAP
                l = fabsf(vpt->SnapLength(vpt->GetCPDisp(m_point, Point3(0,0,1), m_snapPoint, m, TRUE)));
#else
                l = fabsf(vpt->SnapLength(vpt->GetCPDisp(m_point, Point3(0,0,1), m_snapPoint, m)));
#endif
                if(DbgVerify(m_object != nullptr))
                {
                    m_object->SetBox(Box3(Point3(-l, -l, -l), Point3(l, l, l)));
                    m_object->InvalidateMesh();
                    if(msg == MOUSE_POINT) 
                    {
                        m_object->SetCreating(false);

                        if(Length(m - m_snapPoint) < 4)
                            return CREATE_ABORT;
                        else
                            return CREATE_STOP;
                    }
                }
            }
            break;
        }
        break;

    case MOUSE_ABORT:
        if(DbgVerify(m_object != nullptr))
        {
            m_object->SetCreating(false);
        }
        return CREATE_ABORT;
    }

    return CREATE_CONTINUE;
}

// Find a full file path for file.
// If the file is not found, the input is left as-is.
static void FindFullProcPathIfPossible(MaxSDK::AssetManagement::AssetUser &file)
{
    if (!BMMIsFile(file.GetFileName()))
    {
        // Find it if not found already...
        TSTR newfile;
        if (file.GetFullFilePath(newfile))
            file = IAssetManager::GetInstance()->GetAsset(newfile, file.GetType());
    }
}

static AssetUser testFile(TSTR file, AssetEnumCallback &assetEnum, DWORD flags)
{
    AssetUser asset = IAssetManager::GetInstance()->GetAsset(file, kBitmapAsset);
    FindFullProcPathIfPossible(asset);
    if (!(flags & FILE_ENUM_MISSING_ONLY) || !BMMIsFile(asset.GetFileName()))
        assetEnum.RecordAsset(asset);
    return asset;
}

//==============================================================================
// class MAXtoATestObject
//==============================================================================

// The param block descriptor
ParamBlockDesc2 MAXtoATestObject::m_mainPBDesc(
    kMainPBID, _T("Parameters"), IDS_PARAMETERS, &MAXtoATestObject::GetClassDesc(), P_AUTO_CONSTRUCT | P_AUTO_UI_QT | P_MULTIMAP, 
    
    0,
    1,
    0,

    nb_curves, _T("nb_curves"), TYPE_INT, 0, IDS_NB_CURVES,
    p_end,
    
    p_end      
);

ClassDesc2& MAXtoATestObject::GetClassDesc() 
{
    static MAXtoATestObject_ClassDesc classDesc;
    return classDesc;
}

void MAXtoATestObject::InitNodeName(MSTR& s)
{
    s = _M("MAXtoATestObject");
}

MAXtoATestObject::MAXtoATestObject(bool loading)
: m_mainPB(NULL),
  m_creating(false),
  m_recursion(0),
  m_ti(this),
  m_curves(nullptr)
{
    if (!loading)
        GetClassDesc().MakeAutoParamBlocks(this);
}

MAXtoATestObject::~MAXtoATestObject() 
{
    DeleteAllRefs();
}

void MAXtoATestObject::DeleteThis() 
{
    delete this;
}

void MAXtoATestObject::UpdateMesh(TimeValue t) 
{
    if(!m_meshValid.InInterval(t))
        BuildMesh(t);
}

bool MAXtoATestObject::DisplayAsWireframe() 
{
    return true;
}

void MAXtoATestObject::BuildMesh(TimeValue t) 
{
    // Stole this from DummyObject::BuildMesh()
    int nverts = 8;
    int nfaces = 12;
    Point3 va = m_box.pmin;
    Point3 vb = m_box.pmax;
    if(m_box.IsEmpty()) 
    {
        va = Point3(-DD, -DD, -DD);
        vb = Point3( DD,  DD,  DD);
    }

    m_mesh.setNumVerts(nverts);
    m_mesh.setNumFaces(nfaces);

    m_mesh.setVert(0, Point3( va.x, va.y, va.z));
    m_mesh.setVert(1, Point3( vb.x, va.y, va.z));
    m_mesh.setVert(2, Point3( va.x, vb.y, va.z));
    m_mesh.setVert(3, Point3( vb.x, vb.y, va.z));
    m_mesh.setVert(4, Point3( va.x, va.y, vb.z));
    m_mesh.setVert(5, Point3( vb.x, va.y, vb.z));
    m_mesh.setVert(6, Point3( va.x, vb.y, vb.z));
    m_mesh.setVert(7, Point3( vb.x, vb.y, vb.z));

    MakeQuad(&(m_mesh.faces[ 0]), 0,2,3,1,  1);
    MakeQuad(&(m_mesh.faces[ 2]), 2,0,4,6,  2);
    MakeQuad(&(m_mesh.faces[ 4]), 3,2,6,7,  4);
    MakeQuad(&(m_mesh.faces[ 6]), 1,3,7,5,  8);
    MakeQuad(&(m_mesh.faces[ 8]), 0,1,5,4, 16);
    MakeQuad(&(m_mesh.faces[10]), 4,5,7,6, 32);
    m_mesh.InvalidateGeomCache();
    m_mesh.EnableEdgeList(1);

    m_meshValid.SetInfinite();
}

int MAXtoATestObject::Display(TimeValue t, INode* inode, ViewExp *vpt, int flags) 
{
    if ( ! vpt || ! vpt->IsAlive() )
    {
        // why are we here
        DbgAssert(!_T("Doing Display() on invalid viewport!"));
        return FALSE;
    }

    if (MaxSDK::Graphics::IsRetainedModeEnabled())
        return 0;

    GraphicsWindow *gw = vpt->getGW();
    DWORD rlim = gw->getRndLimits();

    Matrix3 mat = inode->GetObjectTM(t);
    gw->setTransform(mat);

    UpdateMesh(t);

    if(DisplayAsWireframe()) 
    {
        gw->setRndLimits(GW_WIREFRAME|GW_EDGES_ONLY|GW_BACKCULL);

        if (inode->Selected())  
        {
            //gw->setColor( LINE_COLOR, GetSelColor());
        }
        else if(!inode->IsFrozen() && !inode->Dependent()) 
        {
            Color color(inode->GetWireColor());
            //gw->setColor( LINE_COLOR, color.r, color.g, color.b);
        }
    }
    
    m_mesh.render(gw, inode->Mtls(), NULL, COMP_ALL, inode->NumMtls());

    gw->setRndLimits(rlim);

    return 0;
}

// from IDisplay
unsigned long MAXtoATestObject::GetObjectDisplayRequirement() const
{
    return 0;
}

bool MAXtoATestObject::PrepareDisplay(const MaxSDK::Graphics::UpdateDisplayContext& prepareDisplayContext)
{
    UpdateMesh(prepareDisplayContext.GetDisplayTime());
    return true;
}

MaxSDK::Graphics::Utilities::MeshEdgeKey GizmoKey;

bool MAXtoATestObject::UpdatePerNodeItems(
                        const MaxSDK::Graphics::UpdateDisplayContext& updateDisplayContext,
                        MaxSDK::Graphics::UpdateNodeContext& nodeContext,
                        MaxSDK::Graphics::IRenderItemContainer& targetRenderItemContainer)
{
    INode* pNode = nodeContext.GetRenderNode().GetMaxNode();
    if (DbgVerify(pNode != nullptr))
    {
        MaxSDK::Graphics::Utilities::MeshEdgeRenderItem* pMeshItem = new MaxSDK::Graphics::Utilities::MeshEdgeRenderItem(&m_mesh, false, false);
        if (pNode->Dependent())
        {
            Color dependentColor = ColorMan()->GetColorAsPoint3(kViewportShowDependencies);
            pMeshItem->SetColor(dependentColor);
        }
        else if (pNode->Selected())
            pMeshItem->SetColor(Color(GetSelColor()));
        else if (pNode->IsFrozen())
            pMeshItem->SetColor(Color(GetFreezeColor()));
        else
        {
            Color color(pNode->GetWireColor());
            pMeshItem->SetColor(Color(color));
        }

        MaxSDK::Graphics::CustomRenderItemHandle tempHandle;
        tempHandle.Initialize();
        tempHandle.SetVisibilityGroup(MaxSDK::Graphics::RenderItemVisible_Gizmo);
        tempHandle.SetCustomImplementation(pMeshItem);
        MaxSDK::Graphics::ConsolidationData data;
        data.Strategy = &MaxSDK::Graphics::Utilities::MeshEdgeConsolidationStrategy::GetInstance();
        data.Key = &GizmoKey;
        tempHandle.SetConsolidationData(data);
        targetRenderItemContainer.AddRenderItem(tempHandle);
        return true;
    }
    else
        return false;
}

CreateMouseCallBack* MAXtoATestObject::GetCreateMouseCallBack() 
{
    MAXtoATestObject_CreateCallback& callback = MAXtoATestObject_CreateCallback::GetInstance();
    callback.SetObject(this);
    return &callback;
}

void MAXtoATestObject::RescaleWorldUnits(float f) 
{
    if (TestAFlag(A_WORK1))
        return;
    SetAFlag(A_WORK1);   

    m_box.pmin *= f;
    m_box.pmax *= f;

    NotifyDependents(FOREVER,PART_ALL,REFMSG_CHANGE);
}

IOResult MAXtoATestObject::Save(ISave *isave) 
{
    ULONG nBytes;
    IOResult res;

    isave->BeginChunk(kChunk_Box);
    res = isave->Write(&m_box, sizeof(m_box), &nBytes);
    isave->EndChunk();

    return res;
}

IOResult MAXtoATestObject::Load(ILoad *iload) 
{
    ULONG nBytes;
    IOResult res = IO_OK;

    while(iload->OpenChunk() == IO_OK) 
    {
        switch(iload->CurChunkID()) 
        {
        case kChunk_Box:
            res = iload->Read(&m_box, sizeof(m_box), &nBytes);
            break;
        }

        iload->CloseChunk();
        if(res != IO_OK)
            return res;
    }

    return IO_OK;
}

void MAXtoATestObject::BeginEditParams(IObjParam *ip, ULONG flags,Animatable *prev) 
{
    ClassDesc2& classDesc = GetClassDesc();
    classDesc.BeginEditParams(ip, this, flags, prev);
}

void MAXtoATestObject::EndEditParams(IObjParam *ip, ULONG flags,Animatable *nex) 
{
    ClassDesc2& classDesc = GetClassDesc();
    classDesc.EndEditParams(ip, this, flags, nex);
}

int MAXtoATestObject::IntersectRay(TimeValue t, Ray& ray, float& at, Point3& norm) 
{
    UpdateMesh(t);
    return m_mesh.IntersectRay(ray, at, norm);
}

void MAXtoATestObject::GetWorldBoundBox(TimeValue t, INode* inode, ViewExp* /*vpt*/, Box3& box ) 
{
    Matrix3 mat = inode->GetObjectTM(t);
    UpdateMesh(t);
    box = m_mesh.getBoundingBox();
    box = box * mat;
}

void MAXtoATestObject::GetLocalBoundBox(TimeValue t, INode* inode, ViewExp* /*vpt*/, Box3& box ) 
{
    UpdateMesh(t);
    box = m_mesh.getBoundingBox();
}

void MAXtoATestObject::GetDeformBBox(TimeValue t, Box3& box, Matrix3 *tm, BOOL useSel ) 
{
    UpdateMesh(t);
    box = m_mesh.getBoundingBox(tm);
}

Mesh* MAXtoATestObject::GetRenderMesh(TimeValue t, INode *inode, View &view, BOOL& needDelete) 
{
    needDelete = FALSE;
    UpdateMesh(t);
    return &m_mesh;
}

Interval MAXtoATestObject::ObjectValidity(TimeValue t) 
{
    Interval valid = FOREVER;
    m_mainPB->GetValidity(t, valid);
    return valid;
}

int MAXtoATestObject::HitTest(TimeValue t, INode* inode, int type, int crossing, int flags, IPoint2 *p, ViewExp *vpt) 
{
    if (MaxSDK::Graphics::IsHardwareHitTesting(vpt))
        return 0;

    if ( ! vpt || ! vpt->IsAlive() )
    {
        // why are we here
        DbgAssert(!_T("Doing Display() on invalid viewport!"));
        return FALSE;
    }
    
    GraphicsWindow* gw = vpt->getGW();
    DWORD rlim = gw->getRndLimits();

    if(DisplayAsWireframe())
        gw->setRndLimits(GW_WIREFRAME|GW_EDGES_ONLY|GW_BACKCULL);

    HitRegion hitRegion;
    MakeHitRegion(hitRegion, type, crossing, 4, p);
    Matrix3 tm = inode->GetObjectTM(t);

    gw->setTransform(tm);
    UpdateMesh(t);

    int res = m_mesh.select(gw, inode->Mtls(), &hitRegion, (flags & HIT_ABORTONHIT), inode->NumMtls());

    gw->setRndLimits(rlim);

    return res;
}

void MAXtoATestObject::Snap(TimeValue t, INode* inode, SnapInfo *snap, IPoint2 *p, ViewExp *vpt) 
{
    if ( ! vpt || ! vpt->IsAlive() )
    {
        // why are we here
        DbgAssert(!_T("Doing Snap() on invalid viewport!"));
        return;
    }

    // Don't snap while creating
    if(m_creating)
        return;

    Matrix3 tm = inode->GetObjectTM(t);	
    GraphicsWindow *gw = vpt->getGW();	
    gw->setTransform(tm);

    UpdateMesh(t);
    m_mesh.snap( gw, snap, p, tm );    
}

const TCHAR* MAXtoATestObject::GetObjectName() 
{
    return L"MAXtoATestObject";
}

RefResult MAXtoATestObject::NotifyRefChanged(const Interval& changeInt, RefTargetHandle hTarget, PartID& partID, RefMessage message, BOOL propagate) 
{
    if (m_recursion > 0) 
        return REF_SUCCEED;
    
    m_recursion++;

    switch (message) 
    {
        case REFMSG_CHANGE:
            if ((hTarget != NULL) && (hTarget->SuperClassID() == PARAMETER_BLOCK2_CLASS_ID)) {
                // Param block changed: invalidate UI

                IParamBlock2* pBlock = static_cast<IParamBlock2*>(hTarget);
                IParamMap2* pMap = pBlock->GetMap();
                int tab = 0;
                ParamID changing_param = pBlock->LastNotifyParamID(tab);
            };
            break;
    }

    m_recursion--;

    return REF_SUCCEED;
}

int MAXtoATestObject::NumRefs() 
{
    return kNumRefs;
}

RefTargetHandle MAXtoATestObject::GetReference(int i) 
{
    switch(i) 
    {
        case kRef_MainPB:
            return m_mainPB;
        default:
            return NULL;
    }
}

void MAXtoATestObject::SetReference(int i, RefTargetHandle rtarg) {

    switch(i) 
    {
        case kRef_MainPB:
            DbgAssert((rtarg == NULL) || (rtarg->SuperClassID() == PARAMETER_BLOCK2_CLASS_ID));
            m_mainPB = static_cast<IParamBlock2*>(rtarg);
            break;
    }
}

RefTargetHandle MAXtoATestObject::Clone(RemapDir &remap) 
{
    MAXtoATestObject* newObject = new MAXtoATestObject(true);

    int count = NumRefs();
    for(int i = 0; i < count; ++i) 
    {
        ReferenceTarget* refTarg = GetReference(i);
        newObject->ReplaceReference(i, remap.CloneRef(refTarg));
    }

    newObject->m_box = m_box;
    newObject->m_mesh = m_mesh;
    newObject->m_meshValid = m_meshValid;
    newObject->m_creating = m_creating;

    BaseClone(this, newObject, remap);
    return newObject;
}

int MAXtoATestObject::NumParamBlocks() 
{
    return 1;
}

IParamBlock2* MAXtoATestObject::GetParamBlock(int i) 
{
    return m_mainPB;
}

IParamBlock2* MAXtoATestObject::GetParamBlockByID(short id) 
{
    return m_mainPB;
}

void MAXtoATestObject::FreeCaches() 
{
    m_meshValid.SetEmpty();
    m_mesh.FreeAll();
}

void MAXtoATestObject::GetClassName(TSTR& s) 
{
    s = L"MAXtoATestObject";
}

SClass_ID MAXtoATestObject::SuperClassID() 
{
    return GEOMOBJECT_CLASS_ID;
}

Class_ID MAXtoATestObject::ClassID() 
{
    return MAXtoATestOBJECT_CLASS_ID;
}

ObjectState MAXtoATestObject::Eval(TimeValue t) 
{
    UpdateMesh(t);
    return ObjectState(this);
}

BaseInterface* MAXtoATestObject::GetInterface(Interface_ID id)
{
    if (id == MAXTOA_TranslationInterface_ID)
        return &(this->m_ti);

    return GeomObject::GetInterface(id);
}

class FlipAxisEnum : public ITreeEnumProc 
{
public:
    Object *obj;
    int     axis;

    virtual int callback(INode * node) 
    {
        Object * ob = node->GetObjectRef();
        if (ob == NULL)
            return TREE_CONTINUE;
        ob = ob->FindBaseObject();
        if (ob == obj) 
        {
            switch (axis)
            {
            case 0:
                node->SetObjOffsetRot(Quat(0.0, 1.0, 0.0, 1.0));
                break;
            case 1:
                node->SetObjOffsetRot(Quat(1.0, 0.0, 0.0, -1.0));
                break;
            case 2:
                node->SetObjOffsetRot(IdentQuat());
                break;
            }
        }
        return TREE_CONTINUE;
    }
};

void MAXtoATestObject::ApplyUpAxis() 
{
    FlipAxisEnum enumFlip;
    enumFlip.obj = this;
    GetCOREInterface7()->GetScene()->EnumTree(&enumFlip);
}

void MAXtoATestObject::GeneratePts(const int n)
{
    const float limit = m_box.pmax.z;
    m_pts.clear();
    m_pts.resize(n * 4);

    for (int i = 0; i < n; ++i)
    {
        for (int j = 0; j < 4; ++j)
        {
            float h = 25.f * static_cast <float> (rand()) / static_cast <float> (RAND_MAX);

            float baseX = (2.f * limit * static_cast <float> (rand()) / static_cast <float> (RAND_MAX)) - limit;
            float baseY = (2.f * limit * static_cast <float> (rand()) / static_cast <float> (RAND_MAX)) - limit;

            int k = i * 4 + j;

            float x = limit + h * j;
            float y = baseX;
            float z = baseY;

            m_pts[k] = Point3(x, y, z);
        }
    }
}

namespace
{
    void MaxToArnold(const Matrix3 &in_m, AtMatrix &out_m)
    {
        out_m[0][0] = in_m[0][0]; out_m[0][1] = in_m[0][1]; out_m[0][2] = in_m[0][2]; out_m[0][3] = 0.0f;
        out_m[1][0] = in_m[1][0]; out_m[1][1] = in_m[1][1]; out_m[1][2] = in_m[1][2]; out_m[1][3] = 0.0f;
        out_m[2][0] = in_m[2][0]; out_m[2][1] = in_m[2][1]; out_m[2][2] = in_m[2][2]; out_m[2][3] = 0.0f;
        out_m[3][0] = in_m[3][0]; out_m[3][1] = in_m[3][1]; out_m[3][2] = in_m[3][2]; out_m[3][3] = 1.0f;
    }
}

Arnold_translation::TranslationInterface_Result MAXtoATestObject_translationInterface::Translate(INode* node,
    Arnold_translation* arnold_translation,
    const TimeValue t,
    Interval& newValidity)
{
    // Translation errors not handled in the demo, alway test "res" for failures;
    Arnold_translation::TranslationInterface_Result res = Arnold_translation::Success;

    arnold_translation->ResetAllOutputs();

    int boxId = 0;
    int curveId = 1;
    int box2Id = 2;

    // 1. A box the same size as the bounding box in the viewport. It uses no flasgs, so all is automatic
    AtNode* b1 = arnold_translation->GetOutput(boxId, AtString("box"), AtString("MyMAXtoATestObjectCube"));
    AiNodeSetVec(b1, AtString("min"), m_obj->m_box.pmin.x, m_obj->m_box.pmin.y, m_obj->m_box.pmin.z);
    AiNodeSetVec(b1, AtString("max"), m_obj->m_box.pmax.x, m_obj->m_box.pmax.y, m_obj->m_box.pmax.z);
    res = arnold_translation->PerformGenericTranslation(boxId, 0, node->GetMtl());


    // 2. Curves on top of the box, does not automatically get the material set. Use the next block to set a material.
    m_obj->m_curves = arnold_translation->GetOutput(curveId, AtString("curves"), AtString("MyMAXtoATestObjectCurves"));
    int flag = Arnold_translation::SkipFlag_Material;
    res = arnold_translation->PerformGenericTranslation(curveId, flag);

    /*
    // 2b. This sets a specific material to the node
    Mtl* mtl = node->GetMtl();
    if (mtl != nullptr)
    {
        AtNode* aMat = arnold_translation->GetNode(Arnold_translation::NodeType_Material, mtl);
        AiNodeSetPtr(m_obj->m_curves, AtString("shader"), aMat);
    }

    */

    // 3. A small box that does not get the transform set (will be at the origin). uncomment the next block to set its transform
    AtNode* b2 = arnold_translation->GetOutput(box2Id, AtString("box"), AtString("MyMAXtoATestObjectCube2"));
    flag = Arnold_translation::SkipFlag_Transform;
    res = arnold_translation->PerformGenericTranslation(box2Id, flag);

    /*
    // 3b. Use thi to set the boxes transform 
    std::vector<Matrix3> transforms = arnold_translation->EvaluateTransforms();

    AtArray* matrices;
    AtMatrix m;

    int nb_matrices = (int)transforms.size();
    matrices = AiArrayAllocate(1, nb_matrices, AI_TYPE_MATRIX);

    int index = 0;
    for (auto it = transforms.begin(); it != transforms.end(); it++, index++)
    {
        MaxToArnold(*it, m);
        AiArraySetMtx(matrices, index, m);
    }

    AiNodeSetArray(b2, "matrix", matrices);
    */    

    // 4. The next block of code handles the deformation keys. It does not take the per object motion blur values. Maybe it should?

    const unsigned int nbDefKeys =  arnold_translation->GetNbDefKeys();
    const int nb_curves = m_obj->m_mainPB->GetInt(MAXtoATestObject::nb_curves, t, 0);

    m_obj->m_nbPoints = AiArrayAllocate(nb_curves, nbDefKeys, AI_TYPE_UINT);
    AiNodeSetArray(m_obj->m_curves, "num_points", m_obj->m_nbPoints);

    m_obj->m_points = AiArrayAllocate(nb_curves * 4, nbDefKeys, AI_TYPE_VECTOR);
    AiNodeSetArray(m_obj->m_curves, "points", m_obj->m_points);

    m_obj->m_curvesRadius = AiArrayAllocate(nb_curves, nbDefKeys, AI_TYPE_FLOAT);
    AiNodeSetArray(m_obj->m_curves, "radius", m_obj->m_curvesRadius);

    if (m_obj->m_pts.size() != nb_curves * 4)
        m_obj->GeneratePts(nb_curves);

    return res;
}

Arnold_translation::TranslationInterface_Result MAXtoATestObject_translationInterface::TranslateKeyframe(INode* node,
    const TimeValue frame_time,
    const TimeValue keyframe_time,
    unsigned int keyframe_index)
{

    const int nb_curves = m_obj->m_mainPB->GetInt(MAXtoATestObject::nb_curves, frame_time, 0);
    int limit = nb_curves * 4;
    int keyIndex = nb_curves * keyframe_index;

    std::vector<unsigned int> nPts;
    nPts.resize(nb_curves);
    std::fill(nPts.begin(), nPts.end(), 4);
    AiArraySetKey(m_obj->m_nbPoints, keyframe_index, nPts.data());

    // simply move the y and z to simulated some keyframe logic.
    std::vector<Point3> pp = m_obj->m_pts;
    for (int i = 0; i < limit; i++)
    {
        pp[i].y += 5.f * keyframe_index;
        pp[i].z += 2.f * keyframe_index;
    }

    AiArraySetKey(m_obj->m_points, keyframe_index, pp.data());

    float rad = 0.25f;
    std::vector<float> rs;
    rs.resize(nb_curves);
    std::fill(rs.begin(), rs.end(), rad);
    AiArraySetKey(m_obj->m_curvesRadius, keyframe_index, rs.data());

    return Arnold_translation::Success;
}
